<?php

namespace App\Filament\App\Resources\AgendaRapatResource\RelationManagers;

use App\Models\AgendaRapat;
use App\Models\User;
use App\Models\Departemen;
use App\Models\Log;
use App\Models\PesertaRapat;
use App\Models\Tamu;
use App\Models\TamuRapat;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;

class TamuRapatRelationManager extends RelationManager
{
    protected static string $relationship = 'TamuRapat';
    protected static ?string $badgeTooltip = 'Jumlah Peserta External';


    public function form(Form $form): Form
    {
        return $form
            ->schema([]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('tamu.name')
                    ->label('Tamu Rapat')
                    ->searchable()
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                $this->TombolSelesaiDanAjukan(),
                $this->TombolTambahPeserta()
            ])
            ->actions([
                Tables\Actions\DeleteAction::make()
                    ->action(function () {
                        $record = $this->ownerRecord;
                        Log::create([
                            'agenda_rapat_id' => $record->id,
                            'user_id' => Auth::id(),
                            'log' => "Peserta rapat dihapus",
                        ]);
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->action(function () {
                            $record = $this->ownerRecord;
                            Log::create([
                                'agenda_rapat_id' => $record->id,
                                'user_id' => Auth::id(),
                                'log' => "Peserta rapat dihapus",
                            ]);
                        }),
                ]),
            ]);
    }

    function TombolTambahPeserta()
    {
        return Tables\Actions\Action::make('Tambah Tamu')
            ->label('Tambah Tamu')
            ->modalWidth('md')
            ->modalHeading('Tambah Tamu Rapat')
            ->modalSubmitActionLabel('Tambah')
            ->extraModalFooterActions(fn(Tables\Actions\Action $action): array => [
                $action->makeModalSubmitAction('Tambah tamu lain', arguments: ['another' => true]),
            ])
            ->form($this->FromTambahPeserta())
            ->action(function (array $data, array $arguments, Tables\Actions\Action $action, Form $form) {
                $this->AksiTambahPeserta($data, $arguments, $action, $form);
            })
            ->hidden(function () {
                $record = $this->ownerRecord;
                if ($record->status == "Draft" || $record->status == "Ditolak") {
                    return false;
                }
                return true;
            });
    }

    function FromTambahPeserta()
    {
        return [
            Forms\Components\Select::make('tamu_id')
                ->options(Tamu::all()->pluck('name', 'id'))
                ->getSearchResultsUsing(function (string $search) {
                    return Tamu::where('name', 'like', "%{$search}%")
                        ->limit(10)
                        ->get()
                        ->mapWithKeys(function (Tamu $tamu) {
                            return [$tamu->id => $tamu->name];
                        })
                        ->toArray();
                })
                ->label('Tamu Rapat')
                ->searchable()
                ->placeholder('Pilih Tamu Rapat')
                ->live()
                ->preload()
                ->optionsLimit(10)
                ->required()
                ->columnSpanFull()
                ->afterStateUpdated(function ($state, callable $set) {
                    if (empty($state)) {
                        return;
                    }
                    $agendaRapat = $this->ownerRecord;

                    // Check for duplicate participants
                    if ($agendaRapat->{$this->getRelationshipName()}()->where('tamu_id', $state)->exists()) {
                        $set('tamu_id', null);
                        Notification::make()
                            ->danger()
                            ->title('Tamu sudah ditambahkan!')
                            ->send();
                        return;
                    }

                    // $agendaHariIni = AgendaRapat::where('tanggal', $agendaRapat->tanggal)->get();

                    // Check for overlapping meetings
                    $overlapping = \App\Models\TamuRapat::where('tamu_id', $state)
                        ->whereHas('agendaRapat', function (Builder $query) use ($agendaRapat) {
                            $query->where('tanggal', $agendaRapat->tanggal)
                                ->where(function (Builder $query) use ($agendaRapat) {
                                    $query->whereBetween('mulai', [$agendaRapat->mulai, $agendaRapat->selesai])
                                        ->orWhereBetween('selesai', [$agendaRapat->mulai, $agendaRapat->selesai])
                                        ->orWhere(function (Builder $query) use ($agendaRapat) {
                                            $query->where('mulai', '<=', $agendaRapat->mulai)
                                                ->where('selesai', '>=', $agendaRapat->selesai);
                                        });
                                });
                        })
                        ->exists();

                    if ($overlapping) {
                        $set('tamu_id', null);
                        Notification::make()
                            ->danger()
                            ->title('Tamu mengikuti rapat lain!')
                            ->body('Tamu sudah terdaftar dalam rapat lain pada waktu tersebut. Silahkan pilih lain.')
                            ->send();
                    }
                }),
        ];
    }
    function AksiTambahPeserta(array $data, array $arguments, Tables\Actions\Action $action, Form $form)
    {
        $record = $this->ownerRecord;

        $jumlahPeserta = PesertaRapat::where('agenda_rapat_id',  $record->id)->count();
        $jumlahTamu = TamuRapat::where('agenda_rapat_id',  $record->id)->count();
        $jumlahOrang = $jumlahPeserta + $jumlahTamu;
        $kuotaRuangan = $record->ruangan->kuota;

        if ($jumlahOrang >= $kuotaRuangan) {
            Notification::make()
                ->danger()
                ->title('Gagal Menambahkan Peserta Rapat')
                ->body('Kuota Ruangan Penuh! kuota ruangan: ' . $kuotaRuangan)
                ->send();
            Log::create([
                'agenda_rapat_id' => $record->id,
                'user_id' => Auth::id(),
                'log' => "Gagal menambahkan peserta rapat: Kuota ruangan penuh!",
            ]);
            return;
        }

        $this->ownerRecord->{$this->getRelationshipName()}()->create([
            'tamu_id' => $data['tamu_id'],
        ]);
        Notification::make()
            ->success()
            ->title('Tamu Rapat Ditambahkan')
            ->body('Tamu rapat berhasil ditambahkan')
            ->send();
        if ($arguments['another'] ?? false) {
            // Reset the form and don't close the modal
            $form->fill();
            $action->halt();
        }
        $record = $this->ownerRecord;
        Log::create([
            'agenda_rapat_id' => $record->id,
            'user_id' => Auth::id(),
            'log' => "Tamu rapat ditambahkan",
        ]);
    }
    function TombolSelesaiDanAjukan()
    {
        return Tables\Actions\Action::make('Selesai & Ajukan')
            ->label('Selesai & Ajukan')
            ->modalHeading('Konfirmasi Pengajuan')
            ->modalWidth('sm')
            ->requiresConfirmation()
            ->action(function () {
                $agendaRapat = $this->ownerRecord;
                $agendaRapat->update(['status' => 'Pengajuan']);
                $record = $this->ownerRecord;
                Log::create([
                    'agenda_rapat_id' => $record->id,
                    'user_id' => Auth::id(),
                    'log' => "Agenda rapat diajukan",
                ]);
                Notification::make()
                    ->success()
                    ->title('Agenda Rapat Diajukan')
                    ->body('Agenda rapat anda akan review oleh admin')
                    ->send();
            })
            ->requiresConfirmation()
            ->color('success')
            ->hidden(function () {
                $record = $this->ownerRecord;
                if ($record->status == "Draft" || $record->status == "Ditolak") {
                    return false;
                }
                return true;
            });
    }
    public static function getBadge(Model $ownerRecord, string $pageClass): ?string
    {
        $jumlah = TamuRapat::where('agenda_rapat_id', $ownerRecord->id)->count();
        return $jumlah;
    }
}
