<?php

namespace App\Filament\App\Resources\AgendaRapatResource\RelationManagers;

use App\Mail\KirimPesan;
use App\Models\User;
use App\Models\Departemen;
use App\Models\Log;
use App\Models\Perwakilan;
use App\Models\Pesan;
use App\Models\PesertaRapat;
use App\Models\Ruangan;
use App\Models\TamuRapat;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Redirect;

class PesertaRapatRelationManager extends RelationManager
{
    protected static string $relationship = 'PesertaRapat';
    protected static ?string $badgeTooltip = 'Jumlah Peserta Internal';


    public function form(Form $form): Form
    {
        return $form
            ->schema([]);
    }

    public function table(Table $table): Table
    {
        $AgendaRapat = $this->ownerRecord;
        $user = User::find(Auth::user()->id);
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('user.name')
                    ->label('Peserta Rapat')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('user.departemen.name')
                    ->label('Departemen')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('user.roles.name')
                    ->label('Jabatan')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn(string $state): string => match ($state) {
                        'Belum Dikonfirmasi' => 'gray',
                        'Diwakilkan'  => 'warning',
                        'Tidak Hadir' => 'danger',
                        'Ada Rapat Lain' => 'danger',
                        'Hadir' => 'success',
                    })
            ])
            ->filters([
                //
            ])
            ->headerActions([
                $this->TombolSelesaiDanAjukan(),
                $this->TombolTambahPeserta($user),
                $this->TombolTambahDepartemen($user)
            ])
            ->actions(self::ActionRecordPesertaRapat($AgendaRapat, $user))
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->action(function () {
                            $record = $this->ownerRecord;
                            Log::create([
                                'agenda_rapat_id' => $record->id,
                                'user_id' => Auth::id(),
                                'log' => "Peserta rapat dihapus",
                            ]);
                        }),
                ]),
            ]);
    }
    protected function ActionRecordPesertaRapat($AgendaRapat, $user): array
    {
        return [
            Tables\Actions\Action::make('Pic')
                ->label('PIC')
                ->visible(function ($record) {
                    return $record->user_id == $this->ownerRecord->pic;
                })
                ->disabled(),
            Tables\Actions\Action::make('Notulen')
                ->label('Notulen')
                ->visible(function ($record) {
                    return $record->user_id == $this->ownerRecord->notulen;
                })
                ->disabled(),

            Tables\Actions\DeleteAction::make()
                ->button()
                ->hidden(function ($record) use ($user) {
                    return $record->user_id == $this->ownerRecord->pic || $record->user_id == $this->ownerRecord->notulen;
                })
                ->action(function ($record) {
                    $record->delete();
                    Log::create([
                        'agenda_rapat_id' => $this->ownerRecord->id,
                        'user_id' => Auth::id(),
                        'log' => "Peserta rapat dihapus",
                    ]);
                }),

            Tables\Actions\ActionGroup::make([
                self::Hadir(),
                self::TidakHadir(),
                self::PerwakilanHadir(),
            ])
                ->hidden(function () use ($user) {
                    if ($user->can('persetujuan_agenda::rapat') && $this->ownerRecord->status != "Draft") {
                        return false;
                    }
                    return true;
                })
                ->dropdownPlacement('bottom-start')
                ->button()
                ->label('Konfirmasi Hadir')
                ->size('sm'),
        ];
    }

    function Hadir()
    {
        return Tables\Actions\Action::make('Hadir')
            ->button()
            ->color('success')
            ->icon('heroicon-c-check')
            ->requiresConfirmation()
            ->action(function ($record) {
                Perwakilan::where('agenda_rapat_id', $record->agenda_rapat_id)->where('user_id', $record->user_id)->delete();
                $record->update([
                    'status' => 'Hadir'
                ]);
                $user = User::find($record->user_id);
                Log::create([
                    'agenda_rapat_id' => $record->agenda_rapat_id,
                    'user_id' => Auth::id(),
                    'log' => "Peserta rapat mengkonfirmasi hadir : " . $user->name,
                ]);
                Notification::make()
                    ->success()
                    ->title('Berhasil!')
                    ->body('Anda sudah mengkonfirmasi untuk hadir ke rapat!')
                    ->send();

                return redirect()->route('filament.app.resources.agenda-rapats.view', ['record' => $record->agenda_rapat_id]);
            });
    }
    function TidakHadir()
    {
        return Tables\Actions\Action::make('Tidak Hadir')
            ->button()
            ->color('danger')
            ->icon('heroicon-c-x-mark')
            ->requiresConfirmation()
            ->action(function ($record) {
                Perwakilan::where('agenda_rapat_id', $record->agenda_rapat_id)->where('user_id', $record->user_id)->delete();
                $record->update([
                    'status' => 'Tidak Hadir'
                ]);
                $user = User::find($record->user_id);
                Log::create([
                    'agenda_rapat_id' => $record->agenda_rapat_id,
                    'user_id' => Auth::id(),
                    'log' => "Peserta rapat mengkonfirmasi absen: " . $user->name,
                ]);
                Notification::make()
                    ->success()
                    ->title('Berhasil!')
                    ->body('Anda sudah mengkonfirmasi untuk tidak hadir ke rapat!')
                    ->send();

                return redirect()->route('filament.app.resources.agenda-rapats.view', ['record' => $record->agenda_rapat_id]);
            });
    }

    function PerwakilanHadir()
    {
        return Tables\Actions\Action::make('Perwakilan')
            ->button()
            ->color('warning')
            ->icon('heroicon-o-user')
            ->modalWidth('md')
            ->form([
                Forms\Components\TextInput::make('name')
                    ->label('Nama Perwakilan')
                    ->required()
                    ->placeholder('Masukkan nama')
                    ->columnSpanFull(),
            ])
            ->action(function (array $data, $record) {
                Perwakilan::where('agenda_rapat_id', $record->agenda_rapat_id)->where('user_id', $record->user_id)->delete();
                $record->update([
                    'status' => 'Diwakilkan'
                ]);
                $user = User::find($record->user_id);
                Perwakilan::create([
                    'agenda_rapat_id' => $record->agenda_rapat_id,
                    'user_id' => $user->id,
                    'name' => $data['name']
                ]);
                Log::create([
                    'agenda_rapat_id' => $record->agenda_rapat_id,
                    'user_id' => Auth::id(),
                    'log' => "Peserta rapat mengkonfirmasi perwakilan: " . $user->name . " oleh :" . $data['name'],
                ]);
                Notification::make()
                    ->success()
                    ->title('Berhasil!')
                    ->body('Anda sudah mengkonfirmasi untuk mewakilkan ke rapat!')
                    ->send();

                return redirect()->route('filament.app.resources.agenda-rapats.view', ['record' => $record->agenda_rapat_id]);
            });
    }

    function TombolSelesaiDanAjukan()
    {
        return Tables\Actions\Action::make('Selesai & Ajukan')
            ->label('Selesai & Ajukan')
            ->modalHeading('Konfirmasi Pengajuan')
            ->modalWidth('sm')
            ->requiresConfirmation()
            ->action(function () {
                $agendaRapat = $this->ownerRecord;
                $agendaRapat->update(['status' => 'Pengajuan']);
                $record = $this->ownerRecord;
                Log::create([
                    'agenda_rapat_id' => $record->id,
                    'user_id' => Auth::id(),
                    'log' => "Agenda rapat diajukan",
                ]);
                Notification::make()
                    ->success()
                    ->title('Agenda Rapat Diajukan')
                    ->body('Agenda rapat anda akan review oleh admin')
                    ->send();
            })
            ->requiresConfirmation()
            ->color('success')
            ->hidden(function () {
                $record = $this->ownerRecord;
                if ($record->status == "Draft" || $record->status == "Ditolak") {
                    return false;
                }
                return true;
            });
    }
    function TombolTambahPeserta($user)
    {
        return Tables\Actions\Action::make('Tambah Peserta')
            ->label('Tambah Peserta')
            ->modalWidth('md')
            ->modalHeading('Tambah Peserta Rapat')
            ->modalSubmitActionLabel('Tambah')
            ->extraModalFooterActions(fn(Tables\Actions\Action $action): array => [
                $action->makeModalSubmitAction('Tambah peserta rapat lain', arguments: ['another' => true]),
            ])
            ->form($this->FromTambahPeserta())
            ->action(function (array $data, array $arguments, Tables\Actions\Action $action, Form $form) {
                $this->AksiTambahPeserta($data, $arguments, $action, $form);
            })
            ->hidden(function () use ($user) {
                $record = $this->ownerRecord;
                if ($record->status == "Pengajuan" && $user->can('persetujuan_agenda::rapat')) {
                    return false;
                } elseif ($record->status == "Draft" || $record->status == "Ditolak") {
                    return false;
                }
                return true;
            });
    }
    function TombolTambahDepartemen($user)
    {
        return Tables\Actions\Action::make('Tambah Departemen')
            ->label('Tambah Departemen')
            ->modalWidth('md')
            ->modalSubmitActionLabel('Tambah')
            ->hidden(function () use ($user) {
                $record = $this->ownerRecord;
                if ($record->status == "Pengajuan" && $user->can('persetujuan_agenda::rapat')) {
                    return false;
                } elseif ($record->status == "Draft" || $record->status == "Ditolak") {
                    return false;
                }
                return true;
            })
            ->action(function (array $data) {
                $this->AksiTambahDepartemen($data);
            })
            ->form($this->FormTambahDepartemen());
    }
    function FromTambahPeserta()
    {
        return [
            Forms\Components\Select::make('user_id')
                ->options(User::all()->pluck('name', 'id'))
                ->getSearchResultsUsing(function (string $search) {
                    return User::query()
                        ->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('pegawai_id', 'like', "%{$search}%")
                        ->orWhere('is_guest', 'like', "%{$search}%")
                        ->orWhereHas('departemen', function (Builder $query) use ($search) {
                            $query->where('name', 'like', "%{$search}%");
                        })
                        ->limit(10)
                        ->get()
                        ->mapWithKeys(function (User $user) {
                            return [$user->id => $user->name];
                        })
                        ->toArray();
                })
                ->label('Peserta Rapat')
                ->searchable()
                ->placeholder('Pilih Peserta Rapat')
                ->live()
                ->preload()
                ->optionsLimit(10)
                ->required()
                ->columnSpanFull()
                ->afterStateUpdated(function ($state, callable $set) {
                    if (empty($state)) {
                        return;
                    }
                    $agendaRapat = $this->ownerRecord;

                    // Check for duplicate participants
                    if ($agendaRapat->{$this->getRelationshipName()}()->where('user_id', $state)->exists()) {
                        $set('user_id', null);
                        Notification::make()
                            ->danger()
                            ->title('Peserta sudah ditambahkan!')
                            ->send();
                        return;
                    }

                    // Check for overlapping meetings
                    $overlapping = \App\Models\PesertaRapat::where('user_id', $state)
                        ->whereHas('agendaRapat', function (Builder $query) use ($agendaRapat) {
                            $query->where('tanggal', $agendaRapat->tanggal)
                                ->where(function (Builder $query) use ($agendaRapat) {
                                    $query->whereBetween('mulai', [$agendaRapat->mulai, $agendaRapat->selesai])
                                        ->orWhereBetween('selesai', [$agendaRapat->mulai, $agendaRapat->selesai])
                                        ->orWhere(function (Builder $query) use ($agendaRapat) {
                                            $query->where('mulai', '<=', $agendaRapat->mulai)
                                                ->where('selesai', '>=', $agendaRapat->selesai);
                                        });
                                });
                        })
                        ->exists();

                    if ($overlapping) {
                        $set('user_id', null);
                        Notification::make()
                            ->danger()
                            ->title('Peserta mengikuti rapat lain!')
                            ->body('Peserta sudah terdaftar dalam rapat lain pada waktu tersebut. Silahkan pilih peserta lain.')
                            ->send();
                    }
                }),
        ];
    }
    function AksiTambahPeserta(array $data, array $arguments, Tables\Actions\Action $action, Form $form)
    {
        $record = $this->ownerRecord;

        $jumlahPeserta = PesertaRapat::where('agenda_rapat_id',  $record->id)->count();
        $jumlahTamu = TamuRapat::where('agenda_rapat_id',  $record->id)->count();
        $jumlahOrang = $jumlahPeserta + $jumlahTamu;
        $kuotaRuangan = $record->ruangan->kuota;

        if ($jumlahOrang >= $kuotaRuangan) {
            Notification::make()
                ->danger()
                ->title('Gagal Menambahkan Peserta Rapat')
                ->body('Kuota Ruangan Penuh! kuota ruangan: ' . $kuotaRuangan)
                ->send();
            Log::create([
                'agenda_rapat_id' => $record->id,
                'user_id' => Auth::id(),
                'log' => "Gagal menambahkan peserta rapat: Kuota ruangan penuh!",
            ]);
            return;
        }

        $this->ownerRecord->{$this->getRelationshipName()}()->create([
            'user_id' => $data['user_id'],
            'status' => 'Belum Dikonfirmasi',
        ]);
        Notification::make()
            ->success()
            ->title('Peserta Rapat Ditambahkan')
            ->body('Peserta rapat berhasil ditambahkan')
            ->send();

        if ($arguments['another'] ?? false) {
            // Reset the form and don't close the modal
            $form->fill();
            $action->halt();
        }

        Log::create([
            'agenda_rapat_id' => $record->id,
            'user_id' => Auth::id(),
            'log' => "Peserta rapat ditambahkan",
        ]);
    }
    function FormTambahdepartemen()
    {
        return [
            Forms\Components\Select::make('departemen_id')
                ->label('Pilih Departemen')
                ->options(Departemen::all()->pluck('name', 'id'))
                ->searchable()
                ->required(),
        ];
    }
    function AksiTambahDepartemen($data)
    {
        $users = User::where('departemen_id', $data['departemen_id'])->get();
        $listUserOverlap = "";
        foreach ($users as $user) {
            // Check for duplicate participants
            if ($this->ownerRecord->{$this->getRelationshipName()}()->where('user_id', $user->id)->exists()) {
                continue;
            }

            //cek kuota ruangan
            $record = $this->ownerRecord;

            $jumlahPeserta = PesertaRapat::where('agenda_rapat_id',  $record->id)->count();
            $jumlahTamu = TamuRapat::where('agenda_rapat_id',  $record->id)->count();
            $jumlahOrang = $jumlahPeserta + $jumlahTamu;
            $kuotaRuangan = $record->ruangan->kuota;

            if ($jumlahOrang >= $kuotaRuangan) {
                Notification::make()
                    ->danger()
                    ->title('Gagal Menambahkan Sebagian Peserta Rapat')
                    ->body('Kuota Ruangan Penuh! kuota ruangan: ' . $kuotaRuangan)
                    ->send();
                Log::create([
                    'agenda_rapat_id' => $record->id,
                    'user_id' => Auth::id(),
                    'log' => "Gagal menambahkan sebagian peserta rapat: Kuota ruangan penuh!",
                ]);
                return;
            }

            // Check for overlapping meetings
            $agendaRapat = $this->ownerRecord;

            $overlapping = \App\Models\PesertaRapat::where('user_id', $user->id)
                ->whereHas('agendaRapat', function (Builder $query) use ($agendaRapat) {
                    $query->where('tanggal', $agendaRapat->tanggal)
                        ->where(function (Builder $query) use ($agendaRapat) {
                            $query->whereBetween('mulai', [$agendaRapat->mulai, $agendaRapat->selesai])
                                ->orWhereBetween('selesai', [$agendaRapat->mulai, $agendaRapat->selesai]);
                            // ->orWhere(function (Builder $query) use ($agendaRapat) {
                            //     $query->where('mulai', '<=', $agendaRapat->mulai)
                            //         ->where('selesai', '>=', $agendaRapat->selesai);
                            // });
                        });
                })
                ->exists();

            if ($overlapping) {
                $listUserOverlap .= $user->name . ", ";
                continue;
            }

            $this->ownerRecord->{$this->getRelationshipName()}()->create([
                'user_id' => $user->id,
                'status' => 'Belum Dikonfirmasi',
            ]);
        }
        if ($listUserOverlap != "") {
            Notification::make()
                ->danger()
                ->title('Peserta mengikuti rapat lain!')
                ->body('Peserta yang tidak bisa ditambahkan: ' . $listUserOverlap)
                ->send();
        } else {
            Notification::make()
                ->success()
                ->title('Peserta Rapat Ditambahkan')
                ->body('Peserta rapat berhasil ditambahkan')
                ->send();
        }
        $record = $this->ownerRecord;
        Log::create([
            'agenda_rapat_id' => $record->id,
            'user_id' => Auth::id(),
            'log' => "Peserta rapat ditambahkan dari departemen" . Departemen::find($data['departemen_id'])->name,
        ]);
    }

    public static function getBadge(Model $ownerRecord, string $pageClass): ?string
    {
        $jumlah = PesertaRapat::where('agenda_rapat_id', $ownerRecord->id)->count();
        return $jumlah;
    }
}
