<?php

namespace App\Filament\App\Resources\AgendaRapatResource\Pages;

use App\Filament\App\Resources\AgendaRapatResource\RelationManagers;
use App\Filament\App\Resources\AgendaRapatResource;
use App\Mail\KirimPesan;
use App\Models\AgendaRapat;
use App\Models\Departemen;
use App\Models\Log;
use App\Models\Pesan;
use App\Models\PesertaRapat;
use App\Models\User;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Exception;
use Filament\Actions;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ViewRecord;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Filament\Forms;

class ViewAgendaRapat extends ViewRecord
{
    protected static string $resource = AgendaRapatResource::class;
    protected function getHeaderActions(): array
    {
        $AgendaRapat = $this->record;
        $user = User::find(Auth::user()->id);
        return [
            $this->TombolUnduhDaftarHadir($AgendaRapat, $user),
            $this->TombolUnduhUndangan($AgendaRapat, $user),
            $this->TombolTerimaPengajuan($AgendaRapat, $user),
            $this->TombolTolakPengajuan($AgendaRapat, $user),
            Actions\EditAction::make('Edit')
                ->button()
                ->color('warning')
                ->icon('heroicon-c-pencil')
                ->visible(function () use ($AgendaRapat, $user) {
                    if ($AgendaRapat->user_id == $user->id && ($AgendaRapat->status == "Draft" || $AgendaRapat->status == "Pengajuan")) {
                        return true;
                    }
                    if ($user->can('persetujuan_agenda::rapat')  && $AgendaRapat->status == "Pengajuan") {
                        return true;
                    }
                    return false;
                }),
            Actions\Action::make('Kembali')
                ->url(function () {
                    return route('filament.app.resources.agenda-rapats.pengajuan');
                })
                ->button()
                ->color('info'),
        ];
    }

    protected function getAllRelationManagers(): array
    {
        return [
            RelationManagers\PesertaRapatRelationManager::class,
            RelationManagers\TamuRapatRelationManager::class,
            RelationManagers\PerwakilanRapatRelationManager::class,
        ];
    }

    function  TombolUnduhDaftarHadir($AgendaRapat, $user)
    {
        return Actions\Action::make('Daftar Hadir')
            ->color('warning')
            ->icon('heroicon-o-document-arrow-down')
            ->hidden(function () use ($AgendaRapat) {
                if ($AgendaRapat->status == "Draft" || $AgendaRapat->status == 'Ditolak' || $AgendaRapat->status == 'Pengajuan') {
                    return true;
                }
                return false;
            })
            ->action(function () use ($AgendaRapat) {

                $rapat = AgendaRapat::where('id', $AgendaRapat->id)
                    ->with('ruangan')
                    ->first();

                $pesertaRapat = $AgendaRapat->PesertaRapat()
                    ->with('user')
                    ->with('user.departemen')
                    ->get();

                $pdf = Pdf::loadView('daftarhadir', ['agendaRapat' => $rapat, 'pesertaRapat' => $pesertaRapat]);
                $pdf->setPaper('A4', 'potrait');
                return response()->streamDownload(function () use ($pdf) {
                    echo $pdf->stream();
                }, 'Daftar hadir rapat ' . $rapat->judul . '.pdf');
            });
    }
    function TombolUnduhUndangan($AgendaRapat, $user)
    {
        return Actions\Action::make('Undangan')
            ->color('warning')
            ->icon('heroicon-o-document-arrow-down')
            ->hidden(function () use ($AgendaRapat) {
                if ($AgendaRapat->status == "Draft" || $AgendaRapat->status == 'Ditolak' || $AgendaRapat->status == 'Pengajuan') {
                    return true;
                }
                return false;
            })
            ->action(function () use ($AgendaRapat) {

                $rapat = AgendaRapat::where('id', $AgendaRapat->id)
                    ->with('ruangan')
                    ->first();

                $pesertaRapat = $AgendaRapat->PesertaRapat()
                    ->with('user')
                    ->with('user.departemen')
                    ->get();

                $creator = $AgendaRapat->creator;
                $departemen = $creator->departemen;

                $pdf = Pdf::loadView('undangan', ['agendaRapat' => $rapat, 'creator' => $creator, 'pesertaRapat' => $pesertaRapat, 'departemen' => $departemen]);
                $pdf->setPaper('A4', 'potrait');
                return response()->streamDownload(function () use ($pdf) {
                    echo $pdf->stream();
                }, 'Undangan rapat.pdf');
            });
    }

    function TombolTerimaPengajuan($AgendaRapat, $user)
    {
        return Actions\Action::make('Terima')
            ->button()
            ->color('success')
            ->icon('heroicon-c-check')
            ->requiresConfirmation()
            ->hidden(function () use ($AgendaRapat, $user) {
                if ($user->can('persetujuan_agenda::rapat') && $AgendaRapat->status == "Pengajuan") {
                    return false;
                }
                return true;
            })
            ->action(function () use ($AgendaRapat) {

                $list = PesertaRapat::where('agenda_rapat_id', $AgendaRapat->id)->get();

                $nomer = "1";
                $agendaRapatTerakhir = AgendaRapat::whereMonth('accepted', Carbon::now()->month)
                    ->whereYear('accepted', Carbon::now()->year)
                    ->orderByDesc('accepted')
                    ->first();

                if ($agendaRapatTerakhir != null) {
                    $nomer = $agendaRapatTerakhir->nomer + 1;
                }
                $AgendaRapat->update([
                    'nomer' => $nomer,
                    'status' => 'Terjadwal',
                    'accepted' => now()
                ]);


                foreach ($list as $item) {

                    //ambil email
                    $email = User::where('id', $item->user_id)->first()->email;

                    //buat pesan
                    Pesan::create([
                        'user_id' => $item->user_id,
                        'type' => 'Konfirmasi',
                        'title' => 'Undangan Rapat',
                        'body' => 'Agenda rapat dengan judul "' . $AgendaRapat->judul . '" telah dijadwalkan. Silakan cek detailnya.',
                        'action' => route('filament.app.resources.agenda-rapats.konfirmasi', [
                            'record' => $AgendaRapat->id,
                        ])
                    ]);

                    try {
                        // kirim email
                        Mail::to($email)->send(new KirimPesan([
                            'type' => 'Konfirmasi',
                            'title' => 'Undangan Rapat',
                            'body' => 'Agenda rapat dengan judul "' . $AgendaRapat->judul . '" telah dijadwalkan. Silakan cek detailnya.',
                            'action' => route('filament.app.resources.agenda-rapats.konfirmasi', [
                                'record' => $AgendaRapat->id,
                            ])
                        ]));
                    } catch (Exception $ex) {
                        continue;
                    }
                }

                Log::create([
                    'agenda_rapat_id' => $AgendaRapat->id,
                    'user_id' => Auth::id(),
                    'log' => "Agenda rapat disetujui",
                ]);

                Notification::make()
                    ->success()
                    ->title('Berhasil menjadwalkan agenda rapat')
                    ->body('Anda menjadwalkan agenda rapat dan mengirim pemberitahuan ke peserta rapat')
                    ->send();

                return redirect()->route('filament.app.resources.agenda-rapats.pengajuan');
            });
    }
    function TombolTolakPengajuan($AgendaRapat, $user)
    {
        return Actions\Action::make('Tolak')
            ->button()
            ->color('danger')
            ->icon('heroicon-c-x-mark')
            ->requiresConfirmation()
            ->hidden(function () use ($AgendaRapat, $user) {

                if ($user->can('persetujuan_agenda::rapat') && $AgendaRapat->status == "Pengajuan") {
                    return false;
                }
                return true;
            })
            ->form([
                Forms\Components\Textarea::make('alasan_ditolak')
                    ->label('Alasan Ditolak')
                    ->required()
                    ->placeholder('Masukkan alasan penolakan agenda rapat')
                    ->columnSpanFull(),
            ])
            ->action(function (array $data) use ($AgendaRapat) {

                $AgendaRapat->update([
                    'status' => 'Ditolak',
                    'alasan_ditolak' => $data['alasan_ditolak']
                ]);
                Log::create([
                    'agenda_rapat_id' => $AgendaRapat->id,
                    'user_id' => Auth::id(),
                    'log' => "Agenda rapat ditolak",
                ]);
                Notification::make()
                    ->danger()
                    ->title('Berhasil menolak agenda rapat')
                    ->send();
                return redirect()->route('filament.app.resources.agenda-rapats.pengajuan');
            });
    }
}
